/*BEGIN_LEGAL 
Intel Open Source License 

Copyright (c) 2002-2013 Intel Corporation. All rights reserved.
 
Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.  Redistributions
in binary form must reproduce the above copyright notice, this list of
conditions and the following disclaimer in the documentation and/or
other materials provided with the distribution.  Neither the name of
the Intel Corporation nor the names of its contributors may be used to
endorse or promote products derived from this software without
specific prior written permission.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE INTEL OR
ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
END_LEGAL */

#include "pin.H"
#include <cassert>
#include <iostream>
#include <fstream>
#include <algorithm>
#include <cstdio>
#include <cstdint>
#include <functional>
#include <stack>
#include <unordered_map>
#include <unordered_set>
#include <vector>

#include <unistd.h>

#include "CallStack.h"

using namespace std;

namespace loops {

void loopTrace(TRACE trace, void *v);
void loopImage(IMG img, void *v);
void loopFini(INT32 code, VOID *v);

}

/* ================================================================== */
// Global variables 
/* ================================================================== */

std::ostream *out = &cout;


/* ===================================================================== */
// Command line switches
/* ===================================================================== */

KNOB<string> outputFile(KNOB_MODE_WRITEONCE,  "pintool",
    "o", "", "File to record tool output");


/* ===================================================================== */
// Utilities
/* ===================================================================== */

void
usage() {
  cerr << "After compiling a program with the loop stubbing utility," << endl
       << "this program identifies loops that do not contain loop" << endl
       << "carried dependencies and examined the potential gain from" << endl
       << "parallelizing them." << endl;
  cerr << KNOB_BASE::StringKnobSummary() << endl;
}


/* ===================================================================== */
// Analysis routines
/* ===================================================================== */

///////////////////////////////////////
// Manage a call stack representation
///////////////////////////////////////

static bool main_entry_seen = false;
static bool prevIpDoesPush = false;
static unordered_set<ADDRINT> pushIps;

const string& target2RtnName(ADDRINT target);
const string& target2LibName(ADDRINT target);

static CallStack callStack(target2RtnName, target2LibName);


void
recordPush(INS ins) {
  pushIps.insert(INS_Address (ins));
}

bool
ipDoesPush(ADDRINT ip) {
  return (pushIps.find(ip) != pushIps.end());
}

const string &
target2RtnName(ADDRINT target) {
  const string & name = RTN_FindNameByAddress(target);

  if (name == "") {
    return *new string("[Unknown routine]");
  } else {
    return *new string(name);
  }
}

const string &
target2LibName(ADDRINT target) {
  PIN_LockClient();

  const RTN rtn = RTN_FindByAddress(target);
  static const string _invalid_rtn("[Unknown image]");

  string name;

  if(RTN_Valid(rtn)) {
    name = IMG_Name(SEC_Img(RTN_Sec(rtn)));
  } else {
    name = _invalid_rtn;
  }

  PIN_UnlockClient();

  return *new string(name);
}


#if defined(TARGET_IA32)  && defined (TARGET_WINDOWS)
static void
processInst(ADDRINT ip) {
  prevIpDoesPush = ipDoesPush(ip);  
}
#endif


static void
enterMainImage(ADDRINT ip, ADDRINT target, ADDRINT sp) {
  main_entry_seen = true;
  callStack.ProcessMainEntry(sp, target);
}


///////////////////////////////////////
// Dependence profiling
///////////////////////////////////////



///////////////////////////////////////
// Entry points for runtime support
///////////////////////////////////////

static void 
handleWrite(uint8_t *ea, UINT32 size, ADDRINT pc) {
}


static void 
handleRead(uint8_t *ea, UINT32 size, ADDRINT pc) {
}


static void
handleCall(ADDRINT ip, ADDRINT target, ADDRINT sp, ADDRINT bp) {
  callStack.ProcessCall(sp, target);

}


static void 
handleReturn(ADDRINT ip, ADDRINT sp) {
  callStack.ProcessReturn(sp, prevIpDoesPush);

}


static void
handleLoopEntry(ADDRINT ip, ADDRINT target, ADDRINT sp) {
}


static void
handleLoopExit(ADDRINT ip, ADDRINT target, ADDRINT sp) {
}


static void
handleLoopLatch(ADDRINT ip, ADDRINT target, ADDRINT sp) {
}


/* ===================================================================== */
// Instrumentation callbacks
/* ===================================================================== */


static void
replaceStub(IMG img, const char *funName, AFUNPTR target) {
  RTN rtn = RTN_FindByName(img, funName);
  if (rtn != RTN_Invalid()) {
    RTN_Open(rtn);
    RTN_InsertCall(rtn, IPOINT_BEFORE, (AFUNPTR)target,
                   IARG_INST_PTR, IARG_ADDRINT, RTN_Address(rtn),
                   IARG_REG_VALUE, REG_STACK_PTR, IARG_END);
    RTN_Close(rtn);
  }
}


void
image(IMG img, void *v) {
  static bool main_rtn_instrumented = false;

  if(!main_rtn_instrumented) {
    RTN rtn = RTN_FindByName(img, "main");
    if(rtn == RTN_Invalid()) {
      rtn = RTN_FindByName(img, "__libc_start_main");      
    }
    // Instrument main
    if(rtn != RTN_Invalid()) {
      main_rtn_instrumented = true;
      RTN_Open(rtn);
      RTN_InsertCall(rtn, IPOINT_BEFORE, (AFUNPTR)enterMainImage,
                     IARG_INST_PTR, IARG_ADDRINT, RTN_Address(rtn),
                     IARG_REG_VALUE, REG_STACK_PTR, IARG_END);
      RTN_Close(rtn);
    }
  }

  // Wrap the calls to loop identification stubs here
  replaceStub(img, "LoOpStUb_entry", (AFUNPTR)handleLoopEntry);
  replaceStub(img, "LoOpStUb_exit",  (AFUNPTR)handleLoopExit);
  replaceStub(img, "LoOpStUb_latch", (AFUNPTR)handleLoopLatch);
}


/*!
 * This function is called every time a new trace is encountered.
 * @param[in]   trace    trace to be instrumented
 * @param[in]   v        value specified by the tool in the TRACE_AddInstrumentFunction
 *                       function call
 */
static void
trace(TRACE trace, void *v) {
  for(BBL bbl = TRACE_BblHead(trace); BBL_Valid(bbl); bbl = BBL_Next(bbl)) {
    INS tail = BBL_InsTail(bbl);

    for(INS ins = BBL_InsHead(bbl); INS_Valid(ins); ins = INS_Next(ins)) {

      UINT32 memOperands = INS_MemoryOperandCount(ins);
      for (UINT32 memOp = 0; memOp < memOperands; memOp++) {
        if (INS_MemoryOperandIsWritten(ins, memOp)) {
          INS_InsertPredicatedCall(
            ins, IPOINT_BEFORE, (AFUNPTR)handleWrite,
            IARG_MEMORYOP_EA, memOp,
            IARG_UINT32, INS_MemoryWriteSize(ins),
            IARG_INST_PTR,
            IARG_END);
        }
        if (INS_MemoryOperandIsRead(ins, memOp)) {
          INS_InsertPredicatedCall(
            ins, IPOINT_BEFORE, (AFUNPTR)handleRead,
            IARG_MEMORYOP_EA, memOp,
            IARG_UINT32, INS_MemoryReadSize(ins),
            IARG_INST_PTR,
            IARG_END);
        }
      }

#if defined(TARGET_IA32)  && defined (TARGET_WINDOWS)
      if (ins != tail) {
        INS_InsertPredicatedCall(ins, IPOINT_BEFORE, (AFUNPTR)processInst,
                                 IARG_INST_PTR, IARG_END);
        if (INS_Opcode(ins)==XED_ICLASS_PUSH) {
          recordPush (ins);
        }
      }
#endif
    }

    if(INS_IsCall(tail)) {
      if(INS_IsDirectBranchOrCall(tail)) {
        ADDRINT target = INS_DirectBranchOrCallTargetAddress(tail);
        INS_InsertPredicatedCall(tail, IPOINT_BEFORE, (AFUNPTR)handleCall,
                                 IARG_INST_PTR, IARG_ADDRINT, target,
                                 IARG_REG_VALUE, REG_STACK_PTR,
                                 IARG_REG_VALUE, REG_EBP, IARG_END);
      } else {
        INS_InsertPredicatedCall(tail, IPOINT_BEFORE, (AFUNPTR)handleCall,
                                 IARG_INST_PTR, IARG_BRANCH_TARGET_ADDR,
                                 IARG_REG_VALUE, REG_STACK_PTR,
                                 IARG_REG_VALUE, REG_EBP, IARG_END);
      }
    }
    if(INS_IsRet(tail)) {
      INS_InsertPredicatedCall(tail, IPOINT_BEFORE, (AFUNPTR)handleReturn,
                               IARG_INST_PTR, IARG_REG_VALUE, REG_STACK_PTR,
                               IARG_END);
    }
  }
}


/*!
 * Print out analysis results.
 * @param[in]   code            exit code of the application
 * @param[in]   v               value specified by the tool in the 
 *                              PIN_AddFiniFunction function call
 */
static void
fini(INT32 code, void *v) {
  *out <<  "===============================================" << endl;
  *out <<  "Finished computing loop information." << endl;
  reportStats();
}


/*!
 * The main procedure of the tool.
 * This function is called when the application image is loaded but not yet started.
 * @param[in]   argc            total number of elements in the argv array
 * @param[in]   argv            array of command line arguments, 
 *                              including pin -t <toolname> -- ...
 */
int
main(int argc, char **argv) {
  // Initialize PIN library. Print help message if -h(elp) is specified
  // in the command line or the command line is invalid 
  PIN_InitSymbols();
  if(PIN_Init(argc, argv)) {
    usage();
    return 0;
  }

  string fileName = outputFile.Value();
  if (!fileName.empty()) {
    out = new std::ofstream(fileName.c_str());
  }

  // Register the loop identification callbacks
  IMG_AddInstrumentFunction(image, 0);
  TRACE_AddInstrumentFunction(trace, 0);
  PIN_AddFiniFunction(fini, 0);

  cerr <<  "===============================================" << endl;
  cerr <<  "Instrumentation successful" << endl;
  if (!fileName.empty()) {
      cerr << "Using " << fileName << " for loop information." << endl;
  }
  cerr <<  "===============================================" << endl;

  // Start the program; never returns
  PIN_StartProgram();
  
  return 0;
}

